/******************************************************************************
* (C) Copyright 2002 by Agilent Technologies, All Rights Reserved.
******************************************************************************/

#include <assert.h>

#include "AgtPCIEControl.h"
#include "AgtOSPort.h"
#include "devpciedefs.h"

#include <time.h>

#pragma region Macros

// This is in milli seconds. Only valid if a wait of 1ms is done after each status read.
#define ISP_TIMEOUT_LONG_MS  1000
#define ISP_TIMEOUT_SHORT_MS  100

// Mask for ISP control register
#define ISP_CONTROL_MASK_REG 0x1F

// Mask for ISP control register function number
#define ISP_CONTROL_MASK_FUNCTION_NUM 0x1E

// Mask for ISP control register lock bit
#define ISP_CONTROL_MASK_LOCK_BIT 0x01

#define ISP_CONTROL_LOCK_BIT_VALUE_INUSE 0x01
#define ISP_CONTROL_LOCK_BIT_VALUE_FREE  0x00

// Check for correct (1-based) port handle
#define HANDLECHECK(handle) {if (handle==AGT_INVALID_PORTHANDLE||handle>MAX_DEVS||OsHandle[handle-1]==(HANDLE)0) AGT_THROW("Invalid port handle passed to function");}

#pragma endregion

#pragma region Statics initialisation

CAgtPCIEControl* CAgtPCIEControl::S_MyOneControllerInstance = NULL;
UInt16 CAgtPCIEControl::S_ReferenceCounter = 0;
CDebugLogger* CAgtPCIEControl::s_pDebugLogger = NULL;

#pragma endregion

#pragma region Constructor and Destructor

CAgtPCIEControl::CAgtPCIEControl
(
  void
)
{
  // Called during construction of PortSelector and Exerciser instance
  for( UInt16 i = 0; i < MAX_DEVS; i++ )
  {
    OsHandle[i] = AGT_INVALID_OSHANDLE;  // from CreateFile()
    mIsGen3Exerciser[i] = 0x0;
    mISAPIEnabled[i] = 0x0;
    mISFPGAEnabled[i] = 0x0;

    m_uFunctionNum[i] = 0x0;
  }

  // Check endianess of current machine
  unsigned char edata[] = { 0x12, 0x34, 0x56, 0x78 };
  isLittleEndian = ( *( ( UInt32* )edata ) == 0x78563412 ); // true on Intel PCs
}

CAgtPCIEControl::~CAgtPCIEControl
(
  void
)
{
}

#pragma endregion

#pragma region Overloaded Operators

void* CAgtPCIEControl::operator new
(
  size_t
)
{
  S_ReferenceCounter++;

  if( S_MyOneControllerInstance == NULL )
  {
    // Create new controller instance
    S_MyOneControllerInstance = ::new CAgtPCIEControl;
    s_pDebugLogger = new CDebugLogger;
    return S_MyOneControllerInstance;
  }
  else
  {
    // Already alive !
    return S_MyOneControllerInstance;
  }
}

void CAgtPCIEControl::operator delete
(
  void*
)
{
  if( S_MyOneControllerInstance == NULL )
  {
    // We are already (or still) dead, do nothing.
    // We usually never get here !
    assert( S_ReferenceCounter == 0 );
  }
  else
  {
    assert( S_ReferenceCounter > 0 );

    S_ReferenceCounter--;

    if( S_ReferenceCounter == 0 )
    {
      // Delete the object
      if( s_pDebugLogger != NULL )
      {
        delete s_pDebugLogger;
        s_pDebugLogger = NULL;
      }

      ::delete( S_MyOneControllerInstance );
      S_MyOneControllerInstance = NULL;
    }
    else
    {
      // Still references to controller in use, so we should not kill us now !
    }
  }
}

#pragma endregion

#pragma region Device Id, Connection related

void CAgtPCIEControl::DeviceIdGet
(
  UInt16 vendId,
  UInt16 devId,
  UInt16 index,
  UInt32* deviceId
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "DeviceIdGet",
    ParamType_UInt16, vendId,
    ParamType_UInt16, devId,
    ParamType_UInt16, index );

  CAgtOSPort::OSDeviceIdGet( vendId, devId, index, deviceId );
}

void CAgtPCIEControl::ConnectPort
(
  UInt32 deviceId,
  AgtPortHandleT *portHandle
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "ConnectPort",
    ParamType_UInt32, deviceId );

  HANDLE osHandle = AGT_INVALID_OSHANDLE;

  if( portHandle == NULL )
  {
    functionLogger.WriteLine( "ConnectPort: NULL pointer (portHandle)." );
    AGT_THROW( "ConnectPort: NULL pointer (portHandle)." );
  }

#ifndef LINUX // deviceId currently ignored under LINUX
  if( deviceId & 0x00f8 ) // Extract Slot number
  {
    // Slotnumber must be always zero in PCIExpress !
    functionLogger.WriteLine( "ConnectPort: Invalid parameter deviceId passed to CAgtPCIEControl::ConnectPort(). DeviceId[7:3] (slotnumber) must always be zero." );
    AGT_THROW( "Invalid parameter deviceId passed to CAgtPCIEControl::ConnectPort(). DeviceId[7:3] (slotnumber) must always be zero." );
  }
#endif

  CAgtOSPort::OSOpenPort( deviceId, osHandle, portHandle ); // portHandle zero based here !

  assert( osHandle ); // In case of failure, OSOpenPort() should have thrown an exception

  if( *portHandle >= MAX_DEVS )
  {
    // We do not support so many devices (must increase MAX_DEVS)
    CAgtOSPort::OSClosePort( osHandle ); // Close device again

    functionLogger.WriteLine( "ConnectPort: Too many devices open. Close any other device first" );
    AGT_THROW( "Too many devices open. Close any other device first" );
  }

  // Store OSHandle for accesses from Read/Write
  OsHandle[*portHandle] = osHandle;

  // For the sake of existing, bad programmed examples,
  // user always works with 1-based port handles.
  ( *portHandle )++;

  HANDLECHECK( *portHandle ); // Should never fail here 

  PortAPICheck( osHandle, *portHandle );
}

void CAgtPCIEControl::PortAPICheck( HANDLE &osHandle, AgtPortHandleT portHandle )
{
  CFunctionLogger functionLogger( s_pDebugLogger, "PortAPICheck", true );

  // Check whether this is U4305
  UInt32 value = 0;

  // We assume that it is Gen3 Exerciser. It will fail in following steps otherwise.

  // CAgtOSPort::OSConfRegDirectRead( osHandle, PCIE_PCIEPORT_REVID_OFFSET * 4, value );

  /*if( ( value & PCIE_PCIEPORT_REVID_MASK ) == PCIE_PCIEPORT_REVID_E2960B_CONDOR ||
      ( value & PCIE_PCIEPORT_REVID_MASK ) == PCIE_PCIEPORT_REVID_E2960B_HIGHLANDER )*/
  //if( ( value & PCIE_PCIEPORT_REVID_MASK ) == PCIE_PCIEPORT_REVID_U4305A_TRISUL )
  //{
  //  mIsGen3Exerciser[portHandle] = 0x1;
  //}
  //else
  //{
  //  mIsGen3Exerciser[portHandle] = 0x0;
  //}

  mIsGen3Exerciser[portHandle] = 0x1;

  GetFunctionLock( portHandle ); // Aquire the function lock

  CAgtOSPort::OSISPRegRead( osHandle, PCIE_CFGPORT_VERSION * 4, value );

  if( ( PCIE_ESW_VERSION_MAJOR != ( ( value >> PCIE_ESW_VERSION_MAJOR_BIT_POS ) & PCIE_ESW_VERSION_MAJOR_MINOR_BITMASK ) ) ||
      ( PCIE_ESW_VERSION_MINOR != ( ( value >> PCIE_ESW_VERSION_MINOR_BIT_POS ) & PCIE_ESW_VERSION_MAJOR_MINOR_BITMASK ) )
    )
  {
    ReleaseFunctionLock( portHandle ); // Release the function lock

    functionLogger.Write( "PortAPICheck: Version number mismatch []" );
    functionLogger.WriteHexUInt( value );
    functionLogger.Write( "]" );
    functionLogger.WriteLineBreak();
    AGT_THROW( "Version number mismatch. The version of the in-system API does not match the version of the ESW." );
  }

  // Check enable bits
  mISAPIEnabled[portHandle]  = ( UInt8 )( value >> PCIE_ESW_VERSION_IS_FOR_API_ENABLE_BIT_POS ) & PCIE_ESW_VERSION_IS_FOR_FPGA_API_ENABLE_BIT_SIZE;
  mISFPGAEnabled[portHandle] = ( UInt8 )( value >> PCIE_ESW_VERSION_IS_FOR_FPGA_ENABLE_BIT_POS ) & PCIE_ESW_VERSION_IS_FOR_FPGA_API_ENABLE_BIT_SIZE;

  if( ( mISAPIEnabled[portHandle] == 0 ) && ( mISFPGAEnabled[portHandle] == 0 ) )
  {
    ReleaseFunctionLock( portHandle ); // Release the function lock

    functionLogger.WriteLine( "PortAPICheck: In-System port cannot be used as the enable bits in the exerciser are not set." );
    AGT_THROW( "In-System port cannot be used as the enable bits in the exerciser are not set." );
  }
  //else
  //{
  //  functionLogger.WriteLine( "PortAPICheck: Device is not Gen3 Exerciser." );
  //  AGT_THROW( "Device is not Gen3 Exerciser." );
  //}

  ReleaseFunctionLock( portHandle ); // Release the function lock
}

void CAgtPCIEControl::OSHandleGet
(
  AgtPortHandleT portHandle,
  HANDLE *osHandle
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "OSHandleGet",
    ParamType_AgtPortHandleT, portHandle );

  HANDLECHECK( portHandle );

  *osHandle = OsHandle[portHandle-1];
}
    
void CAgtPCIEControl::DisconnectPort
(
  AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "DisconnectPort",
    ParamType_AgtPortHandleT, portHandle );

  HANDLECHECK( portHandle );

  // Release FPGA access only if locked by us.
  GetFunctionLock( portHandle );

  // This is done after getting the function lock as this should be modified only if we own the lock.
  FPGAAccessRelease( portHandle );

  ReleaseFunctionLock( portHandle );

  CAgtOSPort::OSClosePort( OsHandle[portHandle-1] );
}

void CAgtPCIEControl::SetFunctionNumber
(
  AgtPortHandleT portHandle,
  UInt8 functionNumber
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "SetFunctionNumber",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt8, functionNumber );

  m_uFunctionNum[portHandle] = functionNumber;
}

#pragma endregion

#pragma region RegRead, RegWrite

// This will do a 'Read' from the 'In-System Port for API'
void CAgtPCIEControl::RegRead
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt8 size,
  UInt32& val
)
{
  RegRead( portHandle, address, size, false, false, val );
}

// This will do a 'Write' from the 'In-System Port for API'
void CAgtPCIEControl::RegWrite
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt8 size,
  UInt32 val
)
{
  RegWrite( portHandle, address, size, false, false, val );
}

// This will do a 'Read' from the 'In-System Port for API'
void CAgtPCIEControl::RegRead
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt8 size,
  bool bDoNotLock,
  bool bDoNotUnlock,
  UInt32& val
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "RegRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address,
    ParamType_UInt8, size,
    ParamType_bool, bDoNotLock,
    ParamType_bool, bDoNotUnlock );

  HANDLECHECK( portHandle );

  if( !bDoNotLock )
    GetFunctionLock( portHandle ); // Aquire the function lock

  // Release the FPGA access. This is done after getting the function lock as this should be
  // modified only if we own the lock.
  if( IsISFPGAEnabled( portHandle ) )
    FPGAAccessRelease( portHandle );

  RegisterRead( portHandle, address, size, val );

  if( !bDoNotUnlock )
    ReleaseFunctionLock( portHandle ); // Release the function lock
}

// This will do a 'Write' from the 'In-System Port for API'
void CAgtPCIEControl::RegWrite
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt8 size,
  bool bDoNotLock,
  bool bDoNotUnlock,
  UInt32 val
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "RegWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address,
    ParamType_UInt8, size,
    ParamType_bool, bDoNotLock,
    ParamType_bool, bDoNotUnlock,
    ParamType_UInt32, val );

  HANDLECHECK( portHandle );

  if( !bDoNotLock )
    GetFunctionLock( portHandle ); // Aquire the function lock

  // Release the FPGA access. This is done after getting the function lock as this should be
  // modified only if we own the lock.
  if( IsISFPGAEnabled( portHandle ) )
    FPGAAccessRelease( portHandle );

  RegisterWrite( portHandle, address, size, val );

  if( !bDoNotUnlock )
    ReleaseFunctionLock( portHandle ); // Release the function lock
}

#pragma endregion

#pragma region RegBlockRead, RegBlockWrite

void CAgtPCIEControl::RegBlockWrite
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32 numBytes,
  UInt32 size,
  UInt8* psaData
)
{
  CFunctionLogger functionLogger(
    s_pDebugLogger, "RegBlockWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address,
    ParamType_UInt32, numBytes,
    ParamType_UInt32, size );

  if( size == 8 )
  {
    for( UInt32 i = 0; i < numBytes; i++ )
    {
      RegWrite( portHandle, address + i, 8, psaData[i] );
    }
  }
  else if( size == 32 )
  {
    UInt32 numBytesBefore = 0;
    UInt32 numDWs = 0;
    UInt32 numBytesAfter = 0;

    numBytesBefore = ( 4 - address % 4 ) % 4;
    if( numBytes < numBytesBefore )
      numBytesBefore = numBytes; // special case
    
    numBytesAfter = ( address + numBytes ) % 4;
    if( numBytesAfter > numBytes )
      numBytesAfter = 0; // special case

    assert( ( numBytes - numBytesBefore - numBytesAfter ) % 4 == 0 );

    numDWs  = ( numBytes - numBytesBefore - numBytesAfter ) / 4;
    
    assert( numBytesBefore + numDWs * 4 + numBytesAfter == numBytes );

    /*
        address numBytes => numBytesBefore numBytesAfter
           0       0              0              0
           0       1              0              1
           1       1              1              0
           2       1              1              0
           3       1              1              0
           4       1              0              1
           5       1              1              0
           0       2              0              2
           1       2              2              0
           2       2              2              0
           3       2              1              1
           4       2              0              2
           0       3              0              3
           1       3              3              0
           2       3              2              1
           3       3              1              2
           4       3              0              3
    */

    // Byte transfer for the (0-3) bytes before the next DW address boundary
    RegBlockWrite( portHandle, address, numBytesBefore, 8, psaData + 0 );

    // DW transfer starting with the first address DW boundary
    assert( ( address + numBytesBefore ) % 4 == 0 );

    for( UInt32 i = 0; i <numDWs; i++ )
    {
      /* Assume psaData={0x78,0x56,0x34,0x12}, i.e 0x78 should goto address 0 in datamemory etc.
         (thats what calling RegWrite() with size=8 four times instead would do).

         Fact is (tried out):
         Writing a DW with value 0x12345678 into datamemory (size=32) at address 0,
         will lead to address 0 containing byte 0x78,..,address 3 containing byte 0x12 (size=8).

         So to write psaData[] correct, we'll have to write a DW value of 0x12345678.
         This corresponds to the memory representation of psaData[] on little
         endian machines, so no byte reordering needed (swap() is a NOP in this case).
         On big endian machines the psaData[] memory layout represents DW value 0x78563412 instead,
         so we additionally need to swap the bytes in order to pass value 0x12345678.
      */
     
      RegWrite( portHandle, address + numBytesBefore + i * 4, 32,
        Swap( *( ( UInt32 * )( psaData + numBytesBefore + i * 4 ) ) ) );
    }

    // Byte transfer for the (0-3) bytes after the last DW address boundary
    RegBlockWrite( portHandle, address + numBytesBefore + numDWs * 4, numBytesAfter, 8,
      psaData + numBytesBefore + numDWs * 4 );
  }
  else
  {
    functionLogger.WriteLine( "RegBlockWrite: Invalid value for parameter size." );
    AGT_THROW( "RegBlockWrite: Invalid value for parameter size." );
  }
}

void CAgtPCIEControl::RegBlockRead
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32 numBytes,
  UInt32 size,
  UInt32* pCount,
  UInt8* psaData
)
{
  CFunctionLogger functionLogger(
    s_pDebugLogger, "RegBlockRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address,
    ParamType_UInt32, numBytes,
    ParamType_UInt32, size );

  UInt32 val = 0;

  if( size == 8 )
  {
    for( UInt32 i = 0; i < numBytes; i++ )
    {
      RegRead( portHandle, address + i, 8, val );
      psaData[i] = ( UInt8 )val;
    }
  }
  else if( size == 32 )
  {
    UInt32 numBytesBefore = 0;
    UInt32 numDWs = 0;
    UInt32 numBytesAfter = 0;
    UInt32 val = 0;

    numBytesBefore = ( 4 - address % 4 ) % 4;
    if( numBytes < numBytesBefore )
      numBytesBefore = numBytes; // special case

    numBytesAfter = ( address + numBytes ) % 4;
    if( numBytesAfter > numBytes )
      numBytesAfter = 0; // special case

    assert( ( numBytes - numBytesBefore - numBytesAfter ) % 4 == 0 );

    numDWs = ( numBytes - numBytesBefore - numBytesAfter ) / 4;
    
    assert( numBytesBefore + numDWs * 4 + numBytesAfter == numBytes );

    // Byte transfer for the (0-3) bytes before the next DW address boundary
    RegBlockRead( portHandle, address, numBytesBefore, 8, NULL, psaData + 0 );

    // DW transfer starting with the first address DW boundary
    assert( ( address + numBytesBefore ) % 4 == 0 );

    for( UInt32 i = 0; i < numDWs; i++ )
    {
      // See discussion in RegBlockWrite()
      RegRead( portHandle, address + numBytesBefore + i * 4, 32, val );
      *( ( UInt32* )( psaData + numBytesBefore + i * 4 ) ) = Swap( val );
    }

    // Byte transfer for the (0-3) bytes after the last DW address boundary
    RegBlockRead( portHandle, address + numBytesBefore + numDWs * 4,
      numBytesAfter, 8, NULL, psaData + numBytesBefore + numDWs * 4 );
  }
  else
  {
    functionLogger.WriteLine( "RegBlockRead: Invalid value for parameter size." );
    AGT_THROW( "RegBlockRead: Invalid value for parameter size." );
  }

  if( pCount )
    *pCount = numBytes;
}

#pragma endregion

#pragma region Config space related

#pragma region ConfRegDirectRead, ConfRegDirectWrite

void CAgtPCIEControl::ConfRegDirectRead
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32& val
)
{
  CFunctionLogger functionLogger(
    s_pDebugLogger, "ConfRegDirectRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address );

  // Executes a direct config access without using
  // the PCI API protocol.
  HANDLECHECK( portHandle );

  CAgtOSPort::OSConfRegDirectRead( OsHandle[portHandle-1], address, val );

  functionLogger.Write( "<< " );
  functionLogger.WriteHexUInt( val );
  functionLogger.WriteLineBreak();
}

void CAgtPCIEControl::ConfRegDirectWrite
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32 val
)
{
  CFunctionLogger functionLogger(
    s_pDebugLogger, "ConfRegDirectWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address,
    ParamType_UInt32, val );

  // Executes a direct config access without using
  // the PCI API protocol.
  HANDLECHECK( portHandle );

  CAgtOSPort::OSConfRegDirectWrite( OsHandle[portHandle-1], address, val );
}

#pragma endregion

#pragma region ConfRegDirectReadWithFnLock, ConfRegDirectWriteWithFnLock

void CAgtPCIEControl::ConfRegDirectReadWithFnLock
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32& val
)
{
  CFunctionLogger functionLogger(
    s_pDebugLogger, "ConfRegDirectReadWithFnLock",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address );

  HANDLECHECK( portHandle );

  // Aquire the function lock
  GetFunctionLock( portHandle );

  CAgtOSPort::OSConfRegDirectRead( OsHandle[portHandle-1], address, val );

  functionLogger.Write( "<< " );
  functionLogger.WriteHexUInt( val );
  functionLogger.WriteLineBreak();

  // Release the function lock
  ReleaseFunctionLock( portHandle );
}

void CAgtPCIEControl::ConfRegDirectWriteWithFnLock
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32 val
)
{
  CFunctionLogger functionLogger(
    s_pDebugLogger, "ConfRegDirectWriteWithFnLock",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address,
    ParamType_UInt32, val );

  HANDLECHECK( portHandle );

  // Aquire the function lock
  GetFunctionLock( portHandle );

  CAgtOSPort::OSConfRegDirectWrite( OsHandle[portHandle-1], address, val );

  // Release the function lock
  ReleaseFunctionLock( portHandle );
}

#pragma endregion

#pragma region Set config space function number and reg read /write

void CAgtPCIEControl::SetConfSpaceFuncNumAndRegRead
(
  AgtPortHandleT portHandle,
  UInt32 functionNum,
  UInt32 address,
  UInt8  size,
  UInt32& val )
{
  CFunctionLogger functionLogger(
    s_pDebugLogger, "SetConfSpaceFuncNumAndRegRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, functionNum,
    ParamType_UInt32, address,
    ParamType_UInt8, size );

  HANDLECHECK( portHandle );

  // Aquire the function lock
  GetFunctionLock( portHandle );

  // Release the FPGA access. This is done after getting the function lock as this should be
  // modified only if we own the lock.
  if( IsISFPGAEnabled( portHandle ) )
    FPGAAccessRelease( portHandle );

  // Set the config space function number
  SetConfSpaceFuncNum( portHandle, functionNum );

  // Do the register read
  RegisterRead( portHandle, address, size, val );

  // Release the function lock
  ReleaseFunctionLock( portHandle );
}

void CAgtPCIEControl::SetConfSpaceFuncNumAndRegWrite
(
  AgtPortHandleT portHandle,
  UInt32 functionNum,
  UInt32 address,
  UInt8  size,
  UInt32 val
)
{
  CFunctionLogger functionLogger(
    s_pDebugLogger, "SetConfSpaceFuncNumAndRegWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, functionNum,
    ParamType_UInt32, address,
    ParamType_UInt8, size,
    ParamType_UInt32, val );

  HANDLECHECK( portHandle );

  // Aquire the function lock
  GetFunctionLock( portHandle );

  // Release the FPGA access. This is done after getting the function lock as this should be
  // modified only if we own the lock.
  if( IsISFPGAEnabled( portHandle ) )
    FPGAAccessRelease( portHandle );

  // Set the config space function number
  SetConfSpaceFuncNum( portHandle, functionNum );

  // Do the register write
  RegisterWrite( portHandle, address, size, val );

  // Release the function lock
  ReleaseFunctionLock( portHandle );
}

#pragma endregion

#pragma endregion

#pragma region ISP using MemRd/MemWr related

void CAgtPCIEControl::ISPRegRead( AgtPortHandleT portHandle, UInt32 address, UInt32& val )
{
  CFunctionLogger functionLogger( s_pDebugLogger, "ISPRegRead", true );

  HANDLECHECK( portHandle );

  CAgtOSPort::OSISPRegRead( OsHandle[portHandle-1], address, val );
}

void CAgtPCIEControl::ISPRegWrite( AgtPortHandleT portHandle, UInt32 address, UInt32 val )
{
  CFunctionLogger functionLogger( s_pDebugLogger, "ISPRegWrite", true );

  HANDLECHECK( portHandle );

  CAgtOSPort::OSISPRegWrite( OsHandle[portHandle-1], address, val );
}

void CAgtPCIEControl::ISPRegReadWithFnLock( AgtPortHandleT portHandle, UInt32 address, UInt32& regVal )
{
  GetFunctionLock( portHandle ); // Aquire the function lock

  ISPRegRead( portHandle, address, regVal );

  ReleaseFunctionLock( portHandle ); // Release the function lock
}

void CAgtPCIEControl::ISPRegWriteWithFnLock( AgtPortHandleT portHandle, UInt32 address, UInt32 regVal )
{
  GetFunctionLock( portHandle ); // Aquire the function lock

  ISPRegWriteVerify( portHandle, address, regVal );

  ReleaseFunctionLock( portHandle ); // Release the function lock
}

void CAgtPCIEControl::ISPRegWriteVerify( AgtPortHandleT portHandle, UInt32 address, UInt32 val )
{
  CFunctionLogger functionLogger( s_pDebugLogger, "ISPRegWriteVerify", true );

  UInt32 timeout = 0;
  UInt32 readVal = 0;
  bool bWriteDone = false;

  while( !bWriteDone &&
         ( timeout < ISP_TIMEOUT_LONG_MS ) )
  {
    if( timeout > 0 )
      WaitMilliSeconds( 1 );

    ISPRegWrite( portHandle, address, val );

    if( timeout > 0 )
      WaitMilliSeconds( 1 );

    ISPRegRead( portHandle, address, readVal );

    bWriteDone = ( val == readVal );
  }

  if( timeout >= ISP_TIMEOUT_LONG_MS )
  {
    functionLogger.Write( "ISPRegWriteVerify: Unable to set FPGA register [Address: " );
    functionLogger.WriteHexUInt( address );
    functionLogger.Write( "{" );
    functionLogger.WriteDecUInt( address );
    functionLogger.Write( "}; Value: " );
    functionLogger.WriteHexUInt( val );
    functionLogger.Write( "{" );
    functionLogger.WriteDecUInt( val );
    functionLogger.Write( "}" );
    functionLogger.WriteLineBreak();
    AGT_THROW( "Unable to set FPGA register" );
  }
}

#pragma endregion

#pragma region In-System port for FPGA access

void CAgtPCIEControl::GetFunctionLockAndFPGAAccess
(
  AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "GetFunctionLockAndFPGAAccess", true );

  if( !IsISFPGAEnabled( portHandle ) )
  {
    functionLogger.WriteLine( "GetFunctionLockAndFPGAAccess: FPGA access is not enabled on this device." );
    AGT_THROW( "FPGA access is not enabled on this device." );
  }

  // Aquire the function lock
  GetFunctionLock( portHandle );
  // Get access to FPGA for ISP.
  FPGAAccessRequest( portHandle );
}

// Function should only be called if we have the function lock
void CAgtPCIEControl::ReleaseFunctionLockAndFPGAAccess
(
  AgtPortHandleT portHandle,
  bool bReleaseFPGA
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "ReleaseFunctionLockAndFPGAAccess", true );

  if( !IsISFPGAEnabled( portHandle ) )
  {
    functionLogger.WriteLine( "ReleaseFunctionLockAndFPGAAccess: FPGA access is not enabled on this device." );
    AGT_THROW( "FPGA access is not enabled on this device." );
  }

  if( bReleaseFPGA )
  {
    // Release access to FPGA for ISP. This is done before releasing the function lock as this
    // should be modified only if we own the lock.
    FPGAAccessRelease( portHandle );
  }
  // Release the function lock
  ReleaseFunctionLock( portHandle );
}

void CAgtPCIEControl::FPGAAccessRequest
(
  AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "FPGAAccessRequest", true );

  HANDLECHECK( portHandle );

  UInt32 val = PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_DEFAULT;
  bool bFPGAAccessGranted = false;
  UInt32 timeout = 0;

  while( !bFPGAAccessGranted &&
         ( timeout < ISP_TIMEOUT_LONG_MS ) )
  {
    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_STATUS_CTRL * 4, val );

    bFPGAAccessGranted = ( ( val & PCIE_CFGPORT_FPGA_STATUS_CTRL_MASK ) == PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_GRANTED );

    if( !bFPGAAccessGranted )
    {
      // We have added a wait between all read/write requests in case of failure in first try in order to avoid
      // packets being missed by FPGA when they are too close to each other.
      if( timeout > 0 )
        WaitMilliSeconds( 1 );

      // Request access
      ISPRegWrite( portHandle, PCIE_CFGPORT_FPGA_STATUS_CTRL * 4, PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_REQUEST );

      if( timeout > 0 )
        WaitMilliSeconds( 1 );
    }

    SanityCheck( portHandle, val );

    timeout++;
  }

  if( timeout >= ISP_TIMEOUT_LONG_MS )
  {
    switch( val )
    {
    case PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_DEFAULT:
      functionLogger.WriteLine( "FPGAAccessRequest: Timeout - FPGA access request failed. Request bit not set (Status: Default)." );
      AGT_THROW( "Timeout: FPGA access request failed. Request bit not set (Status: Default)." );
      break;
    case PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_REQUEST:
      functionLogger.WriteLine( "FPGAAccessRequest: Timeout - FPGA access request failed. ESW did not seem to set grant bit (Status: Request)." );
      AGT_THROW( "Timeout: FPGA access request failed. ESW did not seem to set grant bit (Status: Request)." );
      break;
    case PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_UNKOWN:
      functionLogger.WriteLine( "FPGAAccessRequest: Timeout - FPGA access request failed. Request bit and grant bit both set. (Status: Unknwon)." );
      AGT_THROW( "Timeout: FPGA access request failed. Request bit and grant bit both set. (Status: Unknwon)." );
      break;   
    }
  }
}

void CAgtPCIEControl::FPGAAccessRelease
(
  AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "FPGAAccessRelease", true );

  HANDLECHECK( portHandle );

  UInt32 val = PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_GRANTED;
  bool bFPGAAccessReleased = false;
  UInt32 timeout = 0;

  while ( !bFPGAAccessReleased &&
          ( timeout < ISP_TIMEOUT_LONG_MS ) )
  {
    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_STATUS_CTRL * 4, val );

    bFPGAAccessReleased = ( ( val & PCIE_CFGPORT_FPGA_STATUS_CTRL_MASK ) == PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_DEFAULT );

    if( !bFPGAAccessReleased )
    {
      // We have added a wait between all read/write requests in case of failure in first try in order to avoid
      // packets being missed by FPGA when they are too close to each other.
      if( timeout > 0 )
        WaitMilliSeconds( 1 );

      // Request release
      ISPRegWrite( portHandle, PCIE_CFGPORT_FPGA_STATUS_CTRL * 4, PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_DEFAULT );

      if( timeout > 0 )
        WaitMilliSeconds( 1 );
    }

    SanityCheck( portHandle, val );

    timeout++;
  }

  if( timeout >= ISP_TIMEOUT_LONG_MS )
  {
    switch( val )
    {
    case PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_GRANTED:
      functionLogger.WriteLine( "FPGAAccessRelease: Timeout - FPGA release request failed. Granted bit is still 1 (Status: Granted)." );
      AGT_THROW( "Timeout: FPGA release request failed. Granted bit is still 1 (Status: Granted)." );
      break;
    case PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_REQUEST:
      functionLogger.WriteLine( "FPGAAccessRelease: Timeout - FPGA release request failed. Request bit is still 1 (Status: Request)." );
      AGT_THROW( "Timeout: FPGA release request failed. Request bit is still 1 (Status: Request)." );
      break;
    case PCIE_CFGPORT_FPGA_STATUS_CTRL_VAL_UNKOWN:
      functionLogger.WriteLine( "FPGAAccessRelease: Timeout - FPGA release request failed. Release and grant bit is still 1 (Status: Unknown)." );
      AGT_THROW( "Timeout: FPGA release request failed. Release and grant bit is still 1 (Status: Unknown)." );
      break;
    }
  }
}

void CAgtPCIEControl::FPGAWrite
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32 val
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "FPGAWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address,
    ParamType_UInt32, val );

  HANDLECHECK( portHandle );

  // Clear ack register by writing 'Done' (= 0x1)
  FPGAAckClear( portHandle );

  // Set address
  ISPRegWriteVerify( portHandle, PCIE_CFGPORT_FPGA_ADDRESS * 4, address >> 2 );

  functionLogger.Write( "Address(DW): " );
  functionLogger.WriteHexUInt( address >> 2 ); // DWord
  functionLogger.WriteLineBreak();

  bool bWriteDone = false;
  UInt32 statusVal = PCIE_CFGPORT_FPGA_ACK_STATUS_WAIT;
  UInt32 timeout = 0;

  while ( !bWriteDone &&
          ( timeout < ISP_TIMEOUT_LONG_MS ) )
  {
    if( timeout > 0 )
      WaitMilliSeconds( 1 );

    ISPRegWrite( portHandle, PCIE_CFGPORT_FPGA_VALUE * 4, val );

    if( timeout > 0 )
      WaitMilliSeconds( 1 );

    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_ACK * 4, statusVal );

    bWriteDone = ( ( statusVal & PCIE_CFGPORT_FPGA_ACK_STATUS_MASK ) == PCIE_CFGPORT_FPGA_ACK_STATUS_DONE );

    timeout++;
  }

  if( timeout >= ISP_TIMEOUT_LONG_MS )
  {
    functionLogger.WriteLine( "FPGA Ack not received" );
    AGT_THROW( "FPGA Ack not received" );
  }
}

void CAgtPCIEControl::FPGARead
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32 &val
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "FPGARead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, address );

  HANDLECHECK( portHandle );

  // Clear ack register by writing 'Done' (= 0x1)
  FPGAAckClear( portHandle );

  // Set address
  ISPRegWriteVerify( portHandle, PCIE_CFGPORT_FPGA_ADDRESS * 4, address >> 2 );

  functionLogger.Write( "Address(DW): " );
  functionLogger.WriteHexUInt( address >> 2 ); // DWord
  functionLogger.WriteLineBreak();

  // Read value

  bool bReadDone = false;
  UInt32 readVal = 0;
  UInt32 statusVal = PCIE_CFGPORT_FPGA_ACK_STATUS_WAIT;
  UInt32 timeout = 0;

  while ( !bReadDone &&
          ( timeout < ISP_TIMEOUT_LONG_MS ) )
  {
    if( timeout > 0 )
      WaitMilliSeconds( 1 );

    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_VALUE * 4, readVal );

    if( timeout > 0 )
      WaitMilliSeconds( 1 );

    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_ACK * 4, statusVal );

    bReadDone = ( ( statusVal & PCIE_CFGPORT_FPGA_ACK_STATUS_MASK ) == PCIE_CFGPORT_FPGA_ACK_STATUS_DONE );

    timeout++;
  }

  if( timeout >= ISP_TIMEOUT_LONG_MS )
  {
    functionLogger.WriteLine( "FPGA Ack not received" );
    AGT_THROW( "FPGA Ack not received" );
  }

  val = swapDW( readVal );

  functionLogger.Write( "<< " );
  functionLogger.WriteHexUInt( val ); // Post swap
  functionLogger.Write( "; " );
  functionLogger.Write( "[Pre-Swap: " );
  functionLogger.WriteHexUInt( readVal ); // Pre swap
  functionLogger.Write( "]" );
  functionLogger.WriteLineBreak();
}

void CAgtPCIEControl::FPGAAckClear( AgtPortHandleT portHandle )
{
  bool bAckCleared = false;
  UInt32 statusVal = PCIE_CFGPORT_FPGA_ACK_STATUS_DONE;
  UInt32 timeout = 0;

  while ( !bAckCleared &&
          ( timeout < ISP_TIMEOUT_LONG_MS ) )
  {
    if( timeout > 0 )
      WaitMilliSeconds( 1 );

    ISPRegWrite( portHandle, PCIE_CFGPORT_FPGA_ACK * 4, PCIE_CFGPORT_FPGA_ACK_STATUS_DONE );

    if( timeout > 0 )
      WaitMilliSeconds( 1 );

    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_ACK * 4, statusVal );

    bAckCleared = ( ( statusVal & PCIE_CFGPORT_FPGA_ACK_STATUS_MASK ) == PCIE_CFGPORT_FPGA_ACK_STATUS_WAIT );

    timeout++;
  }

  if( timeout >= ISP_TIMEOUT_LONG_MS )
  {
    AGT_THROW( "Unable to clear ACK" );
  }
}

//void CAgtPCIEControl::ISPValueRegRead( AgtPortHandleT portHandle, UInt32& regVal )
//{
//  CFunctionLogger functionLogger( s_pDebugLogger, "ISPValueRegRead", true );
//
//  bool bReadDone = false;
//  UInt32 readVal = 0;
//  UInt32 val = PCIE_CFGPORT_FPGA_ACK_STATUS_WAIT;
//  UInt32 timeout = 0;
//  UInt32 expectedVal = PCIE_CFGPORT_FPGA_ACK_STATUS_DONE;
//
//  while ( !bReadDone ( val != expectedVal ) &&
//          ( timeout < ISP_TIMEOUT_LONG_MS ) )
//  {
//    if( timeout > 0 )
//    {
//      WaitMilliSeconds( 1 );
//    }
//
//    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_VALUE * 4, readVal );
//    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_ACK * 4, val );
//
//    val &= PCIE_CFGPORT_FPGA_ACK_STATUS_MASK;
//
//    timeout++;
//  }
//
//  if( timeout >= ISP_TIMEOUT_LONG_MS )
//  {
//    functionLogger.WriteLine( "ISPValueRegRead: FPGA Ack not received" );
//    AGT_THROW( "ISPValueRegRead: FPGA Ack not received" );
//  }
//
//  regVal = readVal;
//}

//void CAgtPCIEControl::ISPValueRegWrite( AgtPortHandleT portHandle, UInt32 regVal )
//{
//  CFunctionLogger functionLogger( s_pDebugLogger, "ISPValueRegWrite", true );
//
//  UInt32 val = PCIE_CFGPORT_FPGA_ACK_STATUS_WAIT;
//  UInt32 timeout = 0;
//  UInt32 expectedVal = PCIE_CFGPORT_FPGA_ACK_STATUS_DONE;
//
//  while ( ( val != expectedVal ) &&
//          ( timeout < ISP_TIMEOUT_LONG_MS ) )
//  {
//    if( timeout > 0 )
//    {
//      WaitMilliSeconds( 1 );
//    }
//
//    ISPRegWrite( portHandle, PCIE_CFGPORT_FPGA_VALUE * 4, regVal );
//    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_ACK * 4, val );
//
//    val &= PCIE_CFGPORT_FPGA_ACK_STATUS_MASK;
//
//    timeout++;
//  }
//
//  if( timeout >= ISP_TIMEOUT_LONG_MS )
//  {
//    functionLogger.WriteLine( "ISPValueRegWrite: FPGA Ack not received" );
//    AGT_THROW( "ISPValueRegWrite: FPGA Ack not received" );
//  }
//}

//void CAgtPCIEControl::FPGAWaitAck
//(
//  AgtPortHandleT portHandle,
//  bool waitClear
//)
//{
//  CFunctionLogger functionLogger( s_pDebugLogger, "FPGAWaitAck", true );
//
//  UInt32 val = PCIE_CFGPORT_FPGA_ACK_STATUS_WAIT;
//
//  if( waitClear )
//  {
//    val = PCIE_CFGPORT_FPGA_ACK_STATUS_DONE;
//  }
//
//  UInt32 timeout = 0;
//  UInt32 expectedVal = PCIE_CFGPORT_FPGA_ACK_STATUS_DONE;
//
//  if( waitClear )
//  {
//    expectedVal = PCIE_CFGPORT_FPGA_ACK_STATUS_WAIT;
//  }
//
//  while ( ( val != expectedVal ) &&
//          ( timeout < ISP_TIMEOUT_LONG_MS ) )
//  {
//    if( timeout > 0 )
//    {
//      WaitMilliSeconds( 1 );
//    }
//
//    ISPRegRead( portHandle, PCIE_CFGPORT_FPGA_ACK * 4, val );
//
//    val &= PCIE_CFGPORT_FPGA_ACK_STATUS_MASK;
//
//    timeout++;
//  }
//
//  if( timeout >= ISP_TIMEOUT_LONG_MS )
//  {
//    if( waitClear )
//    {
//      functionLogger.WriteLine( "FPGAWaitAck: Unable to clear ACK" );
//      AGT_THROW( "Unable to clear ACK" );
//    }
//    else
//    {
//      functionLogger.WriteLine( "FPGAWaitAck: FPGA Ack not received" );
//      AGT_THROW( "FPGA Ack not received" );
//    }
//  }
//}

#pragma endregion

#pragma region Helper functions

#pragma region Function locking

void CAgtPCIEControl::GetFunctionLock( AgtPortHandleT portHandle )
{
  CFunctionLogger functionLogger( s_pDebugLogger, "GetFunctionLock", true );

  UInt32 val = 0;
  UInt32 timeout = 0;
  bool bFunctionLockGranted = false;
  HANDLE myOSHandle = OsHandle[portHandle - 1];
  UInt8 uFunctionNum = m_uFunctionNum[portHandle];

  while( !bFunctionLockGranted &&
         ( timeout < ISP_TIMEOUT_LONG_MS ) )
  {
    /*if( timeout > 0 )
      WaitMilliSeconds( 1 );*/

    // Read ISP control register
    CAgtOSPort::OSISPRegRead( myOSHandle, PCIE_CFGPORT_ISP_CONTROL * 4, val );

    if( ( val & ISP_CONTROL_MASK_LOCK_BIT ) == ISP_CONTROL_LOCK_BIT_VALUE_FREE )
    { // Free. Lock it.

      val = uFunctionNum;
      val <<= 1;
      val |= ISP_CONTROL_LOCK_BIT_VALUE_INUSE;

      /*if( timeout > 0 )
        WaitMilliSeconds( 1 );*/

      CAgtOSPort::OSISPRegWrite( myOSHandle, PCIE_CFGPORT_ISP_CONTROL * 4, val );

      /*if( timeout > 0 )
        WaitMilliSeconds( 1 );*/

      CAgtOSPort::OSISPRegRead( myOSHandle, PCIE_CFGPORT_ISP_CONTROL * 4, val );

      if( ( ( ( val & ISP_CONTROL_MASK_FUNCTION_NUM ) >> 1 ) == uFunctionNum ) &&
          ( ( val & ISP_CONTROL_MASK_LOCK_BIT ) == ISP_CONTROL_LOCK_BIT_VALUE_INUSE ) )
      {
        // Granted
        bFunctionLockGranted = true;
      }
    }
    else
    { // Locked. Check if locked by us.
      if( ( ( val & ISP_CONTROL_MASK_FUNCTION_NUM ) >> 1 ) == uFunctionNum )
        bFunctionLockGranted = true;
    }

    if( timeout > 0 )
      WaitMilliSeconds( 1 );

    SanityCheck( myOSHandle, val );

    timeout++;
  }

  if( timeout >= ISP_TIMEOUT_LONG_MS )
  {
    functionLogger.WriteLine( "GetFunctionLock: The function lock cannot be acquired." );
    AGT_THROW( "The function lock cannot be acquired." );
  }
}

void CAgtPCIEControl::ReleaseFunctionLock( AgtPortHandleT portHandle )
{
  CFunctionLogger functionLogger( s_pDebugLogger, "ReleaseFunctionLock", true );

  UInt32 val = 0;
  HANDLE myOSHandle = OsHandle[portHandle - 1];
  UInt8 uFunctionNum = m_uFunctionNum[portHandle];
  bool bFunctionLocked = true;
  UInt32 timeout = 0;

  while( bFunctionLocked  &&
         ( timeout < ISP_TIMEOUT_LONG_MS ) )
  {
    // Read ISP control register
    CAgtOSPort::OSISPRegRead( myOSHandle, PCIE_CFGPORT_ISP_CONTROL * 4, val );
    bFunctionLocked = ( ( val & ISP_CONTROL_MASK_LOCK_BIT ) == ISP_CONTROL_LOCK_BIT_VALUE_INUSE );

    if( bFunctionLocked )
    { // Already locked, so we need to unlock it.
      if( ( ( val & ISP_CONTROL_MASK_FUNCTION_NUM ) >> 1 ) == uFunctionNum )
      { // Locked by us, so we can unlock it.

        val = 0;
        val = uFunctionNum;
        val <<= 1;

        if( timeout > 0 )
          WaitMilliSeconds( 1 );

        // Try to unlock
        CAgtOSPort::OSISPRegWrite( myOSHandle, PCIE_CFGPORT_ISP_CONTROL * 4, val );

        if( timeout > 0 )
          WaitMilliSeconds( 1 );
      }
      else
      {
        // We assume lock has been released as the lock has been acquired by another function.
        bFunctionLocked = false;
      }
    }

    timeout++;
  }

  // We wait for 20ms so that other functions get time to aquire lock. 20ms is used as other
  // functions will retry after 10ms. If the wait time after lock release is 10ms then there
  // is a chance that other functions may miss this window.
  // WaitMilliSeconds( 20 );
}

#pragma endregion

#pragma region In-System Port for API related

void CAgtPCIEControl::RegisterRead( AgtPortHandleT portHandle, UInt32 address, UInt8 size, UInt32 &val )
{
  CFunctionLogger functionLogger( s_pDebugLogger, "RegisterRead",
    ParamType_UInt32, address,
    ParamType_UInt8, size );

  // Prepares status register, sets 'Address' and 'Size'
  RegisterReadWritePrepare( portHandle, address, size );

  UInt32 value = 0;
  UInt32 statusVal = 0;
  bool bReadDone = false;
  bool bHandshakeDone = false;
  UInt32 timeOutShort = 0;
  UInt32 timeOutLong = 0;

  while( !bReadDone &&
         ( timeOutLong < ISP_TIMEOUT_LONG_MS ) )
  {
    if( timeOutLong > 0 )
      WaitMilliSeconds( 1 );

    // Read value to trigger the Port API
    ISPRegRead( portHandle, PCIE_CFGPORT_VALUE * 4, value );

    timeOutShort = 0;

    while( !bHandshakeDone &&
           ( timeOutShort < ISP_TIMEOUT_SHORT_MS ) )
    {
      if( timeOutShort > 0 )
        WaitMilliSeconds( 1 );

      ISPRegRead( portHandle, PCIE_CFGPORT_STATUS_CTRL * 4, statusVal );

      bHandshakeDone = ( ( statusVal & PCIE_CFGPORT_STATUS_CTRL_STATUS_MASK ) == PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_DONE );

      SanityCheck( portHandle, statusVal );

      timeOutShort++;
    }

    if( bHandshakeDone )
      bReadDone = true;

    timeOutLong++;
  }

  if( timeOutLong >= ISP_TIMEOUT_LONG_MS )
  {
    switch( statusVal & PCIE_CFGPORT_STATUS_CTRL_STATUS_MASK )
    {
    case PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_DEFAULT:
      functionLogger.WriteLine( "Timeout - Handshake failed. FPGA did not seem to accept the call (Status: Default)" );
      AGT_THROW( "Timeout: Handshake failed. FPGA did not seem to accept the call (Status: Default)" );
      break;
    case PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_FPGA_ACCEPT:
      functionLogger.WriteLine( "Timeout - Handshake failed. ESW did not seem to get triggered (Status: FPGA Accept)" );
      AGT_THROW( "Timeout: Handshake failed. ESW did not seem to get triggered (Status: FPGA Accept)" );
      break;
    case PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_API_EXECUTE:
      functionLogger.WriteLine( "Timeout - Handshake failed. ESW did not seem to finish the call (Status: API Execute)" );
      AGT_THROW( "Timeout: Handshake failed. ESW did not seem to finish the call (Status: API Execute)" );
      break;
    }
  }

  /*UInt32 errVal = ( val >> PCIE_CFGPORT_STATUS_CTRL_ERRSTATUS_BITPOS ) & PCIE_CFGPORT_STATUS_CTRL_ERRSTATUS_MASK;

  switch( errVal )
  {
  case PCIE_CFGPORT_STATUS_CTRL_ERRSTATUS_ERRVAL_OK:
    break;
  }*/

  // Pick up the real data now
  ISPRegRead( portHandle, PCIE_CFGPORT_VALUE * 4, value );

  functionLogger.Write( "<< " );
  functionLogger.WriteHexUInt( value );
  functionLogger.WriteLineBreak();

  val = value;
}

void CAgtPCIEControl::RegisterWrite( AgtPortHandleT portHandle, UInt32 address, UInt8 size, UInt32 val )
{
  CFunctionLogger functionLogger( s_pDebugLogger, "RegisterWrite",
    ParamType_UInt32, address,
    ParamType_UInt8, size,
    ParamType_UInt32, val );

  // Prepares status register, sets 'Address' and 'Size'
  RegisterReadWritePrepare( portHandle, address, size );

  UInt32 statusVal = 0;
  bool bWriteDone = false;
  bool bHandshakeDone = false;
  UInt32 timeOutShort = 0;
  UInt32 timeOutLong = 0;

  while( !bWriteDone &&
         ( timeOutLong < ISP_TIMEOUT_LONG_MS ) )
  {
    if( timeOutLong > 0 )
      WaitMilliSeconds( 1 );

    // Write value to trigger the Port API
    ISPRegWrite( portHandle, PCIE_CFGPORT_VALUE * 4, val );

    timeOutShort = 0;

    while( !bHandshakeDone &&
           ( timeOutShort < ISP_TIMEOUT_SHORT_MS ) )
    {
      if( timeOutShort > 0 )
        WaitMilliSeconds( 1 );

      ISPRegRead( portHandle, PCIE_CFGPORT_STATUS_CTRL * 4, statusVal );

      bHandshakeDone = ( ( statusVal & PCIE_CFGPORT_STATUS_CTRL_STATUS_MASK ) == PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_DONE );

      SanityCheck( portHandle, statusVal );

      timeOutShort++;
    }

    if( bHandshakeDone )
      bWriteDone = true;

    timeOutLong++;
  }

  if( timeOutLong >= ISP_TIMEOUT_LONG_MS )
  {
    switch( statusVal & PCIE_CFGPORT_STATUS_CTRL_STATUS_MASK )
    {
    case PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_DEFAULT:
      functionLogger.WriteLine( "Timeout - Handshake failed. FPGA did not seem to accept the call (Status: Default)" );
      AGT_THROW( "Timeout: Handshake failed. FPGA did not seem to accept the call (Status: Default)" );
      break;
    case PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_FPGA_ACCEPT:
      functionLogger.WriteLine( "Timeout - Handshake failed. ESW did not seem to get triggered (Status: FPGA Accept)" );
      AGT_THROW( "Timeout: Handshake failed. ESW did not seem to get triggered (Status: FPGA Accept)" );
      break;
    case PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_API_EXECUTE:
      functionLogger.WriteLine( "Timeout - Handshake failed. ESW did not seem to finish the call (Status: API Execute)" );
      AGT_THROW( "Timeout: Handshake failed. ESW did not seem to finish the call (Status: API Execute)" );
      break;
    }
  }

  /*UInt32 errVal = ( val >> PCIE_CFGPORT_STATUS_CTRL_ERRSTATUS_BITPOS ) & PCIE_CFGPORT_STATUS_CTRL_ERRSTATUS_MASK;

  switch( errVal )
  {
  case PCIE_CFGPORT_STATUS_CTRL_ERRSTATUS_ERRVAL_OK:
    break;
  }*/
}

void CAgtPCIEControl::RegisterReadWritePrepare( AgtPortHandleT portHandle, UInt32 address, UInt8 size )
{
  CFunctionLogger functionLogger( s_pDebugLogger, "RegisterReadWritePrepare", true );

  if( IsGen3Exerciser( portHandle ) )
  {
    // Set status control register to default by writing 0x00000000 to the register
    // (this would not set port_busy (bit0) to zero in error case!)
    // (this will set port_busy (bit0) to zero in debugging case!)
    UInt32 statusControlDefaultVal = 0x2;

    // Check if status control is as expected.
    UInt32 statusVal = PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_FPGA_ACCEPT;
    UInt32 timeout = 0;
    bool bDoneSetDefault = false;

    while( !bDoneSetDefault &&
           ( timeout < ISP_TIMEOUT_LONG_MS ) )
    {
      if( timeout > 0 )
        WaitMilliSeconds( 1 );

      ISPRegWrite( portHandle, PCIE_CFGPORT_STATUS_CTRL * 4, PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_DONE );

      if( timeout > 0 )
        WaitMilliSeconds( 1 );

      ISPRegRead( portHandle, PCIE_CFGPORT_STATUS_CTRL * 4, statusVal );

      bDoneSetDefault = ( ( statusVal & PCIE_CFGPORT_STATUS_CTRL_STATUS_MASK ) == PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_DEFAULT );

      SanityCheck( portHandle, statusVal );

      timeout++;
    }

    if( timeout >= ISP_TIMEOUT_LONG_MS )
    {
      functionLogger.WriteLine( "RegisterReadWritePrepare: The PCIE_CFGPORT_STATUS_CTRL cannot be set to default" );
      AGT_THROW( "The PCIE_CFGPORT_STATUS_CTRL cannot be set to default" );
    }

    // Set address
    // !! Note: Removed conditional setting of Address, Size as now ISP may be accessed from
    // multiple functions and not re-writing address/size may cause problems.
    ISPRegWriteVerify( portHandle, PCIE_CFGPORT_ADDRESS * 4, address );

    // Set size
    ISPRegWriteVerify( portHandle, PCIE_CFGPORT_SIZE * 4, size );
  }
}

//void CAgtPCIEControl::HandshakeGen2Gen3(HANDLE myOSHandle)
//{
//  CFunctionLogger functionLogger( s_pDebugLogger, "HandshakeGen2Gen3", true );
//
//  // Start handshake
//  // Make sure the handshake status is set to default before using this function by calling RegisterReadWritePrepare
//  // Make sure the error information is default as well
//  UInt32 val = PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_DEFAULT;
//  UInt32 targetVal = PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_DONE; 
//  UInt32 timeout = 0;
//  
//  while( ( ( val & PCIE_CFGPORT_STATUS_CTRL_STATUS_MASK ) != targetVal ) &&
//         ( timeout < ISP_TIMEOUT_LONG_MS ) )
//  {
//    if( timeout > 0 )
//    {
//      WaitMilliSeconds( 1 );
//    }
//
//    CAgtOSPort::OSISPRegRead( myOSHandle, PCIE_CFGPORT_STATUS_CTRL * 4, val );
//
//    SanityCheck( myOSHandle, val );
//
//    timeout++;
//  }
//
//  if( timeout >= ISP_TIMEOUT_LONG_MS )
//  {
//    switch( val & PCIE_CFGPORT_STATUS_CTRL_STATUS_MASK )
//    {
//    case PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_DEFAULT:
//      functionLogger.WriteLine( "HandshakeGen2Gen3: Timeout - Handshake failed. FPGA did not seem to accept the call (Status: Default)" );
//      AGT_THROW( "Timeout: Handshake failed. FPGA did not seem to accept the call (Status: Default)" );
//      break;
//
//    case PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_FPGA_ACCEPT:
//      functionLogger.WriteLine( "HandshakeGen2Gen3: Timeout - Handshake failed. ESW did not seem to get triggered (Status: FPGA Accept)" );
//      AGT_THROW( "Timeout: Handshake failed. ESW did not seem to get triggered (Status: FPGA Accept)" );
//      break;
//
//    case PCIE_CFGPORT_STATUS_CTRL_STATUS_VAL_API_EXECUTE:
//      functionLogger.WriteLine( "HandshakeGen2Gen3: Timeout - Handshake failed. ESW did not seem to finish the call (Status: API Execute)" );
//      AGT_THROW( "Timeout: Handshake failed. ESW did not seem to finish the call (Status: API Execute)" );
//      break;
//    }
//  }
//
//  UInt32 errVal = ( val >> PCIE_CFGPORT_STATUS_CTRL_ERRSTATUS_BITPOS ) & PCIE_CFGPORT_STATUS_CTRL_ERRSTATUS_MASK;
//
//  switch( errVal )
//  {
//  case PCIE_CFGPORT_STATUS_CTRL_ERRSTATUS_ERRVAL_OK:
//    break;
//  }
//}

#pragma endregion

#pragma region Others

void CAgtPCIEControl::SetConfSpaceFuncNum
(
  AgtPortHandleT portHandle,
  UInt32 functionNum
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "SetConfSpaceFuncNum",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_UInt32, functionNum );

  switch( functionNum )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    RegisterWrite( portHandle, PCIE_CONFIGSPACE_FUNCNUM_A, 32, 0 );
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    RegisterWrite( portHandle, PCIE_CONFIGSPACE_FUNCNUM_B, 32, 1 );
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    RegisterWrite( portHandle, PCIE_CONFIGSPACE_FUNCNUM_C, 32, 2 );
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    RegisterWrite( portHandle, PCIE_CONFIGSPACE_FUNCNUM_D, 32, 3 );
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    RegisterWrite( portHandle, PCIE_CONFIGSPACE_FUNCNUM_E, 32, 4 );
    break;
  }
}

void CAgtPCIEControl::SanityCheck
(
  AgtPortHandleT portHandle,
  UInt32 val
)
{
  HANDLECHECK( portHandle );

  SanityCheck( OsHandle[portHandle - 1], val );
}

void CAgtPCIEControl::SanityCheck
(
  HANDLE myOSHandle,
  UInt32 val
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "SanityCheck", true );

  if( val == 0xffffffff )
  {
    CAgtOSPort::OSISPRegRead( myOSHandle, PCIE_CFGPORT_VERSION * 4, val );
  
    if( PCIE_ESW_VERSION_MAJOR != ( ( val >> PCIE_ESW_VERSION_MAJOR_BIT_POS ) & PCIE_ESW_VERSION_MAJOR_MINOR_BITMASK ) )
    {
      functionLogger.Write( "Sanity check failed; Val: " );
      functionLogger.WriteHexUInt( val );
      functionLogger.Write( " {" );
      functionLogger.WriteDecUInt( val );
      functionLogger.Write( "}" );
      functionLogger.WriteLineBreak();
      AGT_THROW( "Sanity check failed." );
    }
  }
}

void CAgtPCIEControl::WaitMilliSeconds
(
  int msTime
)
{
  CFunctionLogger functionLogger( s_pDebugLogger, "WaitMilliSeconds", true );

#ifdef WIN32
  Sleep( msTime );
#else
  usleep( 1000 * msTime );
#endif

  functionLogger.Write( "*" );

  return;
}

bool CAgtPCIEControl::IsCurMachineLittleEndian
(
  void
) const
{
  return isLittleEndian;
}

bool CAgtPCIEControl::IsISFPGAEnabled
(
  AgtPortHandleT portHandle
) const
{
  HANDLECHECK( portHandle );

  return ( mISFPGAEnabled[portHandle] == 0 ? false : true );
}

bool CAgtPCIEControl::IsGen3Exerciser
(
  AgtPortHandleT portHandle
) const
{
  HANDLECHECK( portHandle );

  return ( mIsGen3Exerciser[portHandle] == 0x1 ? true : false );
}

UInt32 CAgtPCIEControl::Swap
(
  UInt32 val
)
{
  if( !IsCurMachineLittleEndian() )
  {
    // DW endianess swap on big endian machines
    return val >> 24 & 0xff | val >> 8 & 0xff00 | val << 8 & 0xff0000 | val << 24 & 0xff000000;
  }
  else
  {
    return val;
  }
}

UInt32 CAgtPCIEControl::swapDW
(
  const UInt32& val
)
{
  return ( ( val & 0xff ) << 24 ) | ( ( val & 0xff00 ) << 8 ) | ( ( val & 0xff0000 ) >> 8 ) | ( ( val & 0xff000000 ) >> 24 );
}

#pragma endregion

#pragma endregion

#pragma region Debugging

void CAgtPCIEControl::StartDebugLogging
(
  /* [in] */ char* strFileName_p
)
{
  s_pDebugLogger->StartDebugLogging( strFileName_p );
}

void CAgtPCIEControl::StopDebugLogging()
{
  s_pDebugLogger->StopDebugLogging();
}

CDebugLogger* CAgtPCIEControl::GetDebugLogger()
{
  return s_pDebugLogger;
}

#pragma endregion